from dotenv import load_dotenv
load_dotenv()

import os
import requests
import urllib3

# Suppress SSL warnings
urllib3.disable_warnings(urllib3.exceptions.InsecureRequestWarning)

class XUIPanel:
    def __init__(self):
        self.base_url = os.getenv("SERVICE4_PANEL_URL")
        self.username = os.getenv("SERVICE4_PANEL_USERNAME")
        self.password = os.getenv("SERVICE4_PANEL_PASSWORD")
        self.session = requests.Session()
        self.logged_in = False

    def login(self):
        login_url = f"{self.base_url}/login"
        login_data = {
            "referrer": "",
            "username": self.username,
            "password": self.password,
            "login": ""
        }
        headers = {
            "User-Agent": "Mozilla/5.0",
            "Content-Type": "application/x-www-form-urlencoded",
            "Referer": login_url
        }
        resp = self.session.post(login_url, data=login_data, headers=headers, verify=False)
        self.logged_in = resp.status_code == 200
        return self.logged_in

    def ensure_login(self):
        if not self.logged_in:
            return self.login()
        return True

    def get_package_by_id(self, package_id):
        self.ensure_login()
        api_url = f"{self.base_url}/api"
        params = {
            "action": "get_package",
            "package_id": package_id
        }
        resp = self.session.get(api_url, params=params, headers={"X-Requested-With": "XMLHttpRequest"}, verify=False)
        try:
            return resp.json()
        except Exception:
            return None

def brute_force_packages(xui, max_id=50):
    found = []
    print(f"Scanning package IDs 1 to {max_id}...")
    for pid in range(1, max_id+1):
        data = xui.get_package_by_id(pid)
        if data and data.get("result") and data.get("data"):
            pkg = data["data"]
            found.append(pkg)
            print(f"Found: ID={pid}, Full Data: {pkg}")
    if not found:
        print("No packages found in this range.")
    return found

if __name__ == "__main__":
    xui = XUIPanel()
    print("Logging in:", xui.login())
    packages = brute_force_packages(xui, max_id=50)
    print("\nSummary of found packages:")
    for pkg in packages:
        print(
            f"ID: {pkg['id']}, "
            f"Duration: {pkg['official_duration']} {pkg['official_duration_in']}, "
            f"Max Connections: {pkg['max_connections']}, "
            f"Credits: {pkg['cost_credits']}, "
            f"Bouquets: {pkg['bouquets']}"
        )